import { supabase, type Database } from './supabase'
import type { User, Session } from '@supabase/supabase-js'

// Type definitions
export type DbUser = Database['public']['Tables']['users']['Row']
export type DbServer = Database['public']['Tables']['servers']['Row']
export type DbAnalytics = Database['public']['Tables']['analytics']['Row']
export type DbFavorite = Database['public']['Tables']['favorites']['Row']

// Database API class
export class DatabaseAPI {
  // User operations
  static async createUser(userData: {
    username: string
    email: string
    avatar?: string
    country?: string
  }): Promise<DbUser | null> {
    try {
      const { data, error } = await supabase
        .from('users')
        .insert({
          username: userData.username,
          email: userData.email,
          avatar: userData.avatar || `https://api.dicebear.com/7.x/avataaars/svg?seed=${userData.username}`,
          country: userData.country || 'Unknown',
          join_date: new Date().toISOString(),
          last_active: new Date().toISOString()
        })
        .select()
        .single()

      if (error) {
        console.error('Error creating user:', error)
        return null
      }

      return data
    } catch (error) {
      console.error('Database error:', error)
      return null
    }
  }

  static async getUserByEmail(email: string): Promise<DbUser | null> {
    try {
      const { data, error } = await supabase
        .from('users')
        .select('*')
        .eq('email', email)
        .single()

      if (error) {
        console.error('Error fetching user:', error)
        return null
      }

      return data
    } catch (error) {
      console.error('Database error:', error)
      return null
    }
  }

  static async getUserById(id: string): Promise<DbUser | null> {
    try {
      const { data, error } = await supabase
        .from('users')
        .select('*')
        .eq('id', id)
        .single()

      if (error) {
        console.error('Error fetching user:', error)
        return null
      }

      return data
    } catch (error) {
      console.error('Database error:', error)
      return null
    }
  }

  static async updateUser(id: string, updates: Partial<DbUser>): Promise<DbUser | null> {
    try {
      const { data, error } = await supabase
        .from('users')
        .update({
          ...updates,
          updated_at: new Date().toISOString()
        })
        .eq('id', id)
        .select()
        .single()

      if (error) {
        console.error('Error updating user:', error)
        return null
      }

      return data
    } catch (error) {
      console.error('Database error:', error)
      return null
    }
  }

  // Server operations
  static async upsertServer(serverData: Partial<DbServer> & { id: string }): Promise<DbServer | null> {
    try {
      const { data, error } = await supabase
        .from('servers')
        .upsert({
          ...serverData,
          updated_at: new Date().toISOString()
        })
        .select()
        .single()

      if (error) {
        console.error('Error upserting server:', error)
        return null
      }

      return data
    } catch (error) {
      console.error('Database error:', error)
      return null
    }
  }

  static async getServer(id: string): Promise<DbServer | null> {
    try {
      const { data, error } = await supabase
        .from('servers')
        .select('*')
        .eq('id', id)
        .single()

      if (error) {
        console.error('Error fetching server:', error)
        return null
      }

      return data
    } catch (error) {
      console.error('Database error:', error)
      return null
    }
  }

  static async updateServerMetrics(
    serverId: string,
    metrics: {
      clients?: number
      last_ping?: number
      server_load?: number
      memory_usage?: number
      cpu_usage?: number
    }
  ): Promise<boolean> {
    try {
      const { error } = await supabase.rpc('update_server_metrics', {
        server_uuid: serverId,
        new_clients: metrics.clients,
        new_ping: metrics.last_ping,
        new_load: metrics.server_load,
        new_memory: metrics.memory_usage,
        new_cpu: metrics.cpu_usage
      })

      if (error) {
        console.error('Error updating server metrics:', error)
        return false
      }

      return true
    } catch (error) {
      console.error('Database error:', error)
      return false
    }
  }

  // Analytics operations
  static async createAnalytics(analyticsData: {
    user_id: string
    server_id: string
    upvotes?: number
    speed?: number
    duration?: number
    success?: boolean
    type?: 'upvote' | 'powerboost'
  }): Promise<DbAnalytics | null> {
    try {
      const { data, error } = await supabase
        .from('analytics')
        .insert({
          ...analyticsData,
          timestamp: new Date().toISOString()
        })
        .select()
        .single()

      if (error) {
        console.error('Error creating analytics:', error)
        return null
      }

      // Update user's total upvotes if this is an upvote analytics
      if (analyticsData.type === 'upvote' && analyticsData.upvotes) {
        await supabase.rpc('increment_user_upvotes', {
          user_uuid: analyticsData.user_id,
          upvote_amount: analyticsData.upvotes
        })
      }

      return data
    } catch (error) {
      console.error('Database error:', error)
      return null
    }
  }

  static async getUserAnalytics(userId: string, limit: number = 100): Promise<DbAnalytics[]> {
    try {
      const { data, error } = await supabase
        .from('analytics')
        .select('*')
        .eq('user_id', userId)
        .order('created_at', { ascending: false })
        .limit(limit)

      if (error) {
        console.error('Error fetching analytics:', error)
        return []
      }

      return data || []
    } catch (error) {
      console.error('Database error:', error)
      return []
    }
  }

  // Favorites operations
  static async createFavorite(favoriteData: {
    server_id: string
    user_id: string
    name: string
    address: string
    players?: number
    max_players?: number
    ping?: number
  }): Promise<DbFavorite | null> {
    try {
      const { data, error } = await supabase
        .from('favorites')
        .insert({
          ...favoriteData,
          added_at: new Date().toISOString(),
          last_checked: new Date().toISOString(),
          status: 'online'
        })
        .select()
        .single()

      if (error) {
        console.error('Error creating favorite:', error)
        return null
      }

      return data
    } catch (error) {
      console.error('Database error:', error)
      return null
    }
  }

  static async getUserFavorites(userId: string): Promise<DbFavorite[]> {
    try {
      const { data, error } = await supabase
        .from('favorites')
        .select('*')
        .eq('user_id', userId)
        .order('added_at', { ascending: false })

      if (error) {
        console.error('Error fetching favorites:', error)
        return []
      }

      return data || []
    } catch (error) {
      console.error('Database error:', error)
      return []
    }
  }

  static async deleteFavorite(id: string): Promise<boolean> {
    try {
      const { error } = await supabase
        .from('favorites')
        .delete()
        .eq('id', id)

      if (error) {
        console.error('Error deleting favorite:', error)
        return false
      }

      return true
    } catch (error) {
      console.error('Database error:', error)
      return false
    }
  }

  // Real-time subscriptions
  static subscribeToUserAnalytics(userId: string, callback: (payload: any) => void) {
    return supabase
      .channel('user-analytics')
      .on(
        'postgres_changes',
        {
          event: '*',
          schema: 'public',
          table: 'analytics',
          filter: `user_id=eq.${userId}`
        },
        callback
      )
      .subscribe()
  }

  static subscribeToServerUpdates(serverId: string, callback: (payload: any) => void) {
    return supabase
      .channel('server-updates')
      .on(
        'postgres_changes',
        {
          event: 'UPDATE',
          schema: 'public',
          table: 'servers',
          filter: `id=eq.${serverId}`
        },
        callback
      )
      .subscribe()
  }

  // Statistics
  static async getUserStats(userId: string) {
    try {
      const { data, error } = await supabase.rpc('get_user_stats', {
        user_uuid: userId
      })

      if (error) {
        console.error('Error fetching user stats:', error)
        return null
      }

      return data?.[0] || null
    } catch (error) {
      console.error('Database error:', error)
      return null
    }
  }

  // Batch operations
  static async getTopServers(limit: number = 10): Promise<DbServer[]> {
    try {
      const { data, error } = await supabase
        .from('servers')
        .select('*')
        .order('rating', { ascending: false })
        .order('clients', { ascending: false })
        .limit(limit)

      if (error) {
        console.error('Error fetching top servers:', error)
        return []
      }

      return data || []
    } catch (error) {
      console.error('Database error:', error)
      return []
    }
  }

  static async searchServers(query: string, limit: number = 20): Promise<DbServer[]> {
    try {
      const { data, error } = await supabase
        .from('servers')
        .select('*')
        .or(`hostname.ilike.%${query}%,description.ilike.%${query}%,tags.cs.{${query}}`)
        .order('clients', { ascending: false })
        .limit(limit)

      if (error) {
        console.error('Error searching servers:', error)
        return []
      }

      return data || []
    } catch (error) {
      console.error('Database error:', error)
      return []
    }
  }
}

// Authentication helpers
export class AuthAPI {
  static async signUp(email: string, password: string, userData: { username: string }) {
    try {
      const { data, error } = await supabase.auth.signUp({
        email,
        password,
        options: {
          data: {
            username: userData.username
          }
        }
      })

      if (error) {
        console.error('Auth error:', error)
        return { user: null, error }
      }

      // Create user profile in database
      if (data.user) {
        const dbUser = await DatabaseAPI.createUser({
          username: userData.username,
          email: data.user.email!,
          avatar: `https://api.dicebear.com/7.x/avataaars/svg?seed=${userData.username}`
        })

        return { user: data.user, dbUser, error: null }
      }

      return { user: data.user, error: null }
    } catch (error) {
      console.error('Auth error:', error)
      return { user: null, error }
    }
  }

  static async signIn(email: string, password: string) {
    try {
      const { data, error } = await supabase.auth.signInWithPassword({
        email,
        password
      })

      if (error) {
        console.error('Auth error:', error)
        return { user: null, session: null, error }
      }

      // Get user profile from database
      let dbUser = null
      if (data.user) {
        dbUser = await DatabaseAPI.getUserByEmail(data.user.email!)
      }

      return { user: data.user, session: data.session, dbUser, error: null }
    } catch (error) {
      console.error('Auth error:', error)
      return { user: null, session: null, error }
    }
  }

  static async signOut() {
    try {
      const { error } = await supabase.auth.signOut()
      return { error }
    } catch (error) {
      console.error('Auth error:', error)
      return { error }
    }
  }

  static async getCurrentUser() {
    try {
      const { data: { user }, error } = await supabase.auth.getUser()
      
      if (error || !user) {
        return { user: null, dbUser: null, error }
      }

      const dbUser = await DatabaseAPI.getUserByEmail(user.email!)
      return { user, dbUser, error: null }
    } catch (error) {
      console.error('Auth error:', error)
      return { user: null, dbUser: null, error }
    }
  }

  static onAuthStateChange(callback: (event: string, session: Session | null) => void) {
    return supabase.auth.onAuthStateChange(callback)
  }
}
