import { toast } from '@/hooks/use-toast'

export interface AppError {
  code: string
  message: string
  details?: any
  timestamp: Date
}

export class ErrorHandler {
  private static errors: AppError[] = []

  static handle(error: any, context?: string): AppError {
    const appError: AppError = {
      code: error.code || 'UNKNOWN_ERROR',
      message: error.message || 'An unknown error occurred',
      details: error.details || error,
      timestamp: new Date()
    }

    // Log error
    console.error(`[${context || 'APP'}] Error:`, appError)

    // Store error for debugging
    this.errors.push(appError)
    
    // Keep only last 100 errors
    if (this.errors.length > 100) {
      this.errors = this.errors.slice(-100)
    }

    // Show user-friendly toast
    this.showUserError(appError, context)

    return appError
  }

  private static showUserError(error: AppError, context?: string) {
    let title = 'Error'
    let description = error.message

    // Customize error messages based on error codes
    switch (error.code) {
      case 'PGRST301':
        title = 'Authentication Required'
        description = 'Please log in to continue'
        break
      case 'PGRST116':
        title = 'Access Denied'
        description = 'You do not have permission to perform this action'
        break
      case 'PGRST202':
        title = 'Not Found'
        description = 'The requested resource was not found'
        break
      case 'auth/invalid-email':
        title = 'Invalid Email'
        description = 'Please enter a valid email address'
        break
      case 'auth/weak-password':
        title = 'Weak Password'
        description = 'Password should be at least 6 characters'
        break
      case 'auth/email-already-in-use':
        title = 'Email Already Registered'
        description = 'This email is already associated with an account'
        break
      case 'auth/user-not-found':
        title = 'User Not Found'
        description = 'No account found with this email address'
        break
      case 'auth/wrong-password':
        title = 'Invalid Password'
        description = 'The password you entered is incorrect'
        break
      case 'NETWORK_ERROR':
        title = 'Connection Error'
        description = 'Please check your internet connection and try again'
        break
      case 'SERVER_ERROR':
        title = 'Server Error'
        description = 'Something went wrong on our end. Please try again later'
        break
      default:
        if (context) {
          title = `${context} Error`
        }
    }

    toast({
      title,
      description,
      variant: 'destructive'
    })
  }

  static getRecentErrors(count: number = 10): AppError[] {
    return this.errors.slice(-count).reverse()
  }

  static clearErrors(): void {
    this.errors = []
  }

  static exportErrors(): string {
    return JSON.stringify(this.errors, null, 2)
  }
}

// Utility functions for common error scenarios
export const handleSupabaseError = (error: any, context?: string) => {
  return ErrorHandler.handle(error, context || 'Database')
}

export const handleNetworkError = (error: any, context?: string) => {
  const networkError = {
    ...error,
    code: 'NETWORK_ERROR'
  }
  return ErrorHandler.handle(networkError, context || 'Network')
}

export const handleValidationError = (message: string, context?: string) => {
  const validationError = {
    code: 'VALIDATION_ERROR',
    message
  }
  return ErrorHandler.handle(validationError, context || 'Validation')
}
